<?php
require_once('api_regiones.php');
require_once('api_comunas.php');
require_once('api_sucursales.php');
require_once('api_normalizacion.php');
require_once('api_tarificacion.php');
require_once('tbl_envios.php');

function correoschile_iniciar_shipping_method() {
    if (!class_exists('CorreosChile_Shipping_Method')) {
        class CorreosChile_Shipping_Method extends WC_Shipping_Method {
            public function __construct() {
                $this->id = 'correoschile_shipping_method';
                $this->method_title = __('Correos de Chile', 'correoschile_shipping_method');
                $this->method_description = __('Método de envío de Correos de Chile', 'correoschile_shipping_method');
                $this->availability = 'including';
                $this->countries = array('CL');
                $this->init();
                $this->enabled = isset($this->settings['enabled']) ? $this->settings['enabled'] : 'yes';
                $this->title = isset( $this->settings['title'] ) ? $this->settings['title'] : __('Correos de Chile', 'correoschile_shipping_method');
            }
            function init() {
                $this->init_form_fields();
                $this->init_settings();
                add_action('woocommerce_update_options_shipping_' . $this->id, array($this,'process_admin_options'));
            }
            function init_form_fields() { 
                $this->form_fields = array(
                    'enabled' => array(
                    'title' => __('Activo', 'correoschile_shipping_method' ),
                    'type' => 'checkbox',
                    'description' => __('Activar el método de envío.', 'correoschile_shipping_method' ),
                    'default' => 'yes'
                    ),
                    'title' => array(
                        'title' => __('Nombre', 'correoschile_shipping_method' ),
                        'type' => 'text',
                        'description' => __('Este es el nombre que aparecerá para el usuario.', 'correoschile_shipping_method' ),
                        'default' => __('Envíos Nacionales', 'correoschile_shipping_method')
                    )
                );
            }
            
            public function calculate_shipping($package = array()) {
                if (!in_array( 'dokan-lite/dokan.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {
                    update_option('correoschile_calcularPorTienda', 'N','yes');
                }
                $tipoCalculo = filter_input(INPUT_POST, 'shipping_calculator');
                if(is_null($tipoCalculo)) {
                    $postData = filter_input(INPUT_POST, 'post_data');
                    if(isset($postData)){
                        $data = correoschile_queryStringToArray($postData);
                        $tipoCalculo = $data['shipping_calculator'];
                    }
                }
                $total = 0;
                switch($tipoCalculo){
                    case 'carrito': $total = $this->calcular_envio_desde_carrito($package); break;
                    case 'finalizar_compra': $total = $this->calcular_envio_desde_finalizar_compra($package, $data); break;
                    default: $total = $this->calcular_envio($package); break;
                }
                
                $rate = ['id' => $this->id, 'label' => $this->title, 'cost' => $this->marginar_costo_envio($total) ];                
                $this->add_rate($rate);
            }
            
            function marginar_costo_envio($total){
                if($total === 0){
                    return $total;
                }
                $marginar = get_option('correoschile_marginar', '');
                $valor = get_option('correoschile_marginar_valor', 0);
                if(is_numeric($valor) === false){
                    $valor = 0;
                }
                if($marginar !== ''){
                    switch($marginar){
                        case 'P': //Porcentaje
                            if($valor < 0)
                                $valor = 100;
                            $total = ($total / 100) * $valor;
                            break;
                        case 'V': //Valor
                            $total = $total + intval($valor);
                            break;
                    }
                }
                if($total < 0) {
                    $total = 0;
                }
                return $total;
            }
            
            function calcular_envio_desde_carrito($package) {
                $input = filter_input(INPUT_POST, 'calc_shipping_tipoDespacho');
                $tipoDespacho = is_null($input) || is_bool($input) ? '24' : trim($input);
                $pais = $package['destination']['country'];
                $region = $package['destination']['state'];
                $comuna = $package['destination']['city'];
                $input = filter_input(INPUT_POST, 'calc_shipping_direccion');
                $direccion = is_null($input) || is_bool($input) ? '' : trim($input);
                $codPostal = $package['destination']['postcode'];
                
                if(!direccion !== '' && $comuna !== '') {
                    try {
                        $direccionNormalizada = correoschile_normalizacion($direccion, $comuna);
                        $codPostal = $direccionNormalizada->CodigoPostal;
                        if(tipoDespacho === '24'){
                            $direccion = $direccionNormalizada->Calle . ' ' . $direccionNormalizada->Numero;
                            $direccion = trim($direccion);
                        }
                    } catch (Exception $e){
                        //nada
                    }
                }
                
                $comunaRemitente = get_option('correoschile_comuna');
                $total = 0;
                $tarificar = get_option('correoschile_tarificar', '');
                if($tarificar === 'S') {
                    $no_tarificar = get_option('correoschile_no_tarificar', 0);
                    $no_tarificar = intval($no_tarificar);
                    $total_carrito =  0;
                    foreach ($package['contents'] as $item_id => $values ) { 
                        $producto = $values['data'];
                        $total_carrito += $producto->get_price() * $values['quantity'];
                    }
                    if($no_tarificar === 0 || $total_carrito <= $no_tarificar){
                        $esValido = $this->correoschile_validarCalculoCoberturaPorProducto($comunaRemitente, $comuna, $tipoDespacho, $package);
                        if($esValido === true) {
                            $calcularPorTienda = get_option('correoschile_calcularPorTienda', '');
                            if($calcularPorTienda === 'S') {
                                $total = $this->correoschile_calcular_coberturaPorProductoPorTienda($comunaRemitente, $tipoDespacho, $package);
                            } else {
                                $total = $this->correoschile_calcular_coberturaPorProducto($comunaRemitente, $comuna, $tipoDespacho, $package);
                            }
                        }
                    }
                }
                return $total;

                WC()->session->set('correoschile_shipping_tipoDespacho', $tipoDespacho);
                WC()->session->set('correoschile_shipping_direccion', $direccion);
                WC()->session->set('chosen_shipping_methods', array($this->id));
                WC()->customer->set_shipping_country($pais);
                WC()->customer->set_shipping_state($region);
                WC()->customer->set_shipping_city($comuna);
                WC()->customer->set_shipping_address($direccion);
                WC()->customer->set_shipping_address_1($direccion);
                WC()->customer->set_shipping_postcode($codPostal);
                
                $ship_to_destination = get_option('woocommerce_ship_to_destination', '');
                if(get_option('woocommerce_ship_to_destination') === 'billing_only'){
                    WC()->session->set('correoschile_billing_tipoDespacho', $tipoDespacho);
                    WC()->session->set('correoschile_billing_direccion', $direccion);
                    WC()->customer->set_billing_country($pais);
                    WC()->customer->set_billing_state($region);
                    WC()->customer->set_billing_city($comuna);
                    WC()->customer->set_billing_address($direccion);
                    WC()->customer->set_billing_address_1($direccion);
                    WC()->customer->set_billing_postcode($codPostal);                        
                }
                return $total;
            }
            
            /**
             * Calcula de costos de envío cuando se actualiza vía post la finalización de la compra.
             * Adicionalmante actualiza los parámetros propios de correos de chile (tipoDespacho).
             * 
             * @param array $package Información de los productos del carrito.
             * @param array $data    Cambios realizados en el formulario de finalización de la compra vía post.
             * @return int           Costo total del envío. 
             */
            function calcular_envio_desde_finalizar_compra($package, $data){
                $ultimo_cambio = $data['ultimo_cambio'];
                $comuna = '';
                $tipoDespacho = '';
                switch($ultimo_cambio)
                {
                    case 'billing_state':
                        $tipoDespacho = $data['billing_tipoDespacho'];
                        WC()->session->set('correoschile_billing_tipoDespacho', $tipoDespacho);
                        break;
                    case 'billing_tipoDespacho':
                    case 'billing_comuna':
                        $tipoDespacho = $data['billing_tipoDespacho'];
                        WC()->session->set('correoschile_billing_tipoDespacho', $tipoDespacho);
                        $comunaObtenida = correoschile_obtenerComuna($data['billing_state'], $data['billing_comuna']);
                        $comuna = $comunaObtenida->NombreComuna;
                        break;
                    case 'shipping_state':
                        $tipoDespacho = $data['shipping_tipoDespacho'];
                        WC()->session->set('correoschile_shipping_tipoDespacho', $tipoDespacho);
                        break;
                    case 'shipping_tipoDespacho':
                    case 'shipping_comuna': 
                        $tipoDespacho = $data['shipping_tipoDespacho'];
                        WC()->session->set('correoschile_shipping_tipoDespacho', $tipoDespacho);
                        $comunaObtenida = correoschile_obtenerComuna($data['shipping_state'], $data['shipping_comuna']);
                        $comuna = $comunaObtenida->NombreComuna;
                        break;
                    default:
                        $comuna = $package['destination']['city'];
                        $tipoDespacho = WC()->session->get('correoschile_shipping_tipoDespacho');
                        if(!isset($tipoDespacho) || $tipoDespacho === '') $tipoDespacho = '24';
                        WC()->session->set('correoschile_billing_tipoDespacho', $tipoDespacho);
                        $ship_to_destination = get_option('woocommerce_ship_to_destination', '');
                        if(get_option('woocommerce_ship_to_destination') === 'billing_only'){
                            $tipoDespacho = WC()->session->get('correoschile_billing_tipoDespacho');
                            if(!isset($tipoDespacho) || $tipoDespacho === '') $tipoDespacho = '24';
                            WC()->session->set('correoschile_shipping_tipoDespacho', $tipoDespacho);
                        }
                        break;
                }
                
                $comunaRemitente = get_option('correoschile_comuna');
                $total = 0;
                $tarificar = get_option('correoschile_tarificar', '');
                if($tarificar === 'S') {
                    $no_tarificar = get_option('correoschile_no_tarificar', 0);
                    $no_tarificar = intval($no_tarificar);
                    $total_carrito =  0;
                    foreach ($package['contents'] as $item_id => $values ) { 
                        $producto = $values['data'];
                        $total_carrito += $producto->get_price() * $values['quantity'];
                    }
                    if($no_tarificar === 0 || $total_carrito <= $no_tarificar){
                        $esValido = $this->correoschile_validarCalculoCoberturaPorProducto($comunaRemitente, $comuna, $tipoDespacho, $package, true);
                        if($esValido === true) {
                            $calcularPorTienda = get_option('correoschile_calcularPorTienda', '');
                            if($calcularPorTienda === 'S') {
                                $total = $this->correoschile_calcular_coberturaPorProductoPorTienda($comunaRemitente, $tipoDespacho, $package);
                            } else {
                                $total = $this->correoschile_calcular_coberturaPorProducto($comunaRemitente, $comuna, $tipoDespacho, $package);
                            }
                        }
                    }
                }
                return $total;
            }
            
            /**
             * Calcula de costos de envío (Método por defecto).
             * 
             * @param type $package Información de los productos del carrito.
             * @return int          Costo total del envío.
             */
            function calcular_envio($package) {
                $tipoDespacho = WC()->session->get('correoschile_shipping_tipoDespacho');
                if(!isset($tipoDespacho) || $tipoDespacho === '') $tipoDespacho = '24';
                WC()->session->set('correoschile_billing_tipoDespacho', $tipoDespacho);
                $ship_to_destination = get_option('woocommerce_ship_to_destination', '');
                if(get_option('woocommerce_ship_to_destination') === 'billing_only'){
                    $tipoDespacho = WC()->session->get('correoschile_billing_tipoDespacho');
                    if(!isset($tipoDespacho) || $tipoDespacho === '') $tipoDespacho = '24';
                    WC()->session->set('correoschile_shipping_tipoDespacho', $tipoDespacho);
                }
                
                $comunaRemitente = get_option('correoschile_comuna');
                $comunaDestinatario = $package['destination']['city'];
                $total = 0;
                $tarificar = get_option('correoschile_tarificar', '');
                if($tarificar === 'S') {
                    $no_tarificar = get_option('correoschile_no_tarificar', 0);
                    $no_tarificar = intval($no_tarificar);
                    $total_carrito =  0;
                    foreach ($package['contents'] as $item_id => $values ) { 
                        $producto = $values['data'];
                        $total_carrito += $producto->get_price() * $values['quantity'];
                    }
                    if($no_tarificar === 0 || $total_carrito <= $no_tarificar){
                        $esValido = $this->correoschile_validarCalculoCoberturaPorProducto($comunaRemitente, $comunaDestinatario, $tipoDespacho, $package);
                        if($esValido === true) {
                            $calcularPorTienda = get_option('correoschile_calcularPorTienda', '');
                            if($calcularPorTienda === 'S') {
                                $total = $this->correoschile_calcular_coberturaPorProductoPorTienda($comunaRemitente, $tipoDespacho, $package);
                            } else {
                                $total = $this->correoschile_calcular_coberturaPorProducto($comunaRemitente, $comunaDestinatario, $tipoDespacho, $package);
                            }
                        }
                    }
                }
                return $total;
            }
            
            function correoschile_validarCalculoCoberturaPorProducto($comunaRemitente, $comunaDestinatario, $tipoDespacho, $package, $mostrarMensajes = false){
                $esValido = true;
                $notice = '';

                //DETECTAR QUE EL METODO ELEGIDO ES CORREOS CHILE
                $chosen_methods = WC()->session->get('chosen_shipping_methods');
                if(is_array($chosen_methods) && in_array('correoschile_shipping_method', $chosen_methods)) {
                    if(!isset($comunaRemitente) || $comunaRemitente === '') {
                        $notice .=  ($notice !== '' ? '<br/>' : '') . 'La comuna de la tienda no se encuentra definida.';
                        $esValido = false;
                    }                
                    if(!isset($comunaDestinatario) || $comunaDestinatario === '') {
                        $notice .= ($notice !== '' ? '<br/>' : '') . 'Debe seleccionar una comuna para calcular el envío.';
                        $esValido = false;
                    }
                    if(!isset($tipoDespacho) || $tipoDespacho === '') {
                        $notice .=  ($notice !== '' ? '<br/>' : '') . 'Debe seleccionar el tipo de despacho para calcular el envío.';
                        $esValido = false;
                    }
                    foreach ($package['contents'] as $item_id => $values ) { 
                        $producto = $values['data'];
                        $peso = is_numeric($producto->get_weight()) ? wc_get_weight($producto->get_weight(), 'kg') : 0;
                        $width = is_numeric($producto->get_width()) ? $producto->get_width() : 0;
                        $length = is_numeric($producto->get_length()) ? $producto->get_length() : 0;
                        $height = is_numeric($producto->get_height()) ? $producto->get_height() : 0;
                    
                        $volumen = wc_get_dimension($width * $length * $height, 'cm');
                        if($peso === 0 || $volumen === 0){
                            $notice .=  ($notice !== '' ? '<br/>' : '') . 'El producto "' . $producto->get_name() . '" no tiene su peso o dimensiones definidos.';
                            $esValido = false;
                        } else {
                            if($volumen > 0) {
                                $volumen = $volumen / 1000000;
                            }
                        } 
                        if($peso > 50 || $volumen > 1) {
                            $notice .=  ($notice !== '' ? '<br/>' : '') . 'El producto "' . $producto->get_name() . '" excede el peso o dimensiones aceptadas por nuestros servicios de paquetería.';
                            $esValido = false;
                        }
                    }
                    if ($mostrarMensajes === true) {
                        $notices = wc_get_notices();
                        if($notice !== ''){
                            $notice = '<strong>CorreosChile informa:</strong><br/>' . $notice;
                            if(!wc_has_notice($notice, 'notice')){
                                wc_add_notice($notice, 'notice');
                            }
                        }
                    }
                    return $esValido;
                }
            }
            
            /**
             * Calcula el costo de envio de los productos en base a la comuna, tipo de despacho, peso y volumen.
             * Opcionalmente, prepara un mensaje de validación cuando los parametros de entrada son incorrectos.
             * En caso de error del servicio, prepara un mensaje de error informando lo ocurrido.
             * 
             * @param string $comuna       Nombre de la comuna de destino.
             * @param string $tipoDespacho Identificador del tipo de despacho. (24: Despacho a domicilio, 07: Despacho a Sucursal)
             * @param array $package       Información de los productos del carrito.
             * @param bool $validar        Indica si se realizan validaciones.
             * @return int
             */
            function correoschile_calcular_coberturaPorProducto($comunaRemitente, $comunaDestinatario, $tipoDespacho, $package){
                $error = '';
                $total = 0;
                $peso = 0;
                $volumen = 0;
                foreach ($package['contents'] as $item_id => $values ) { 
                    $producto = $values['data'];
                    $peso = $peso + (wc_get_weight($producto->get_weight(), 'kg') * $values['quantity']);
                    $volumen = $volumen + ((wc_get_dimension($producto->get_width() * $producto->get_length() * $producto->get_height(), 'cm')/1000000) * $values['quantity']);
                }
                try {
                    $coberturaPorProducto = correoschile_coberturaPorProducto($tipoDespacho, $comunaRemitente, $comunaDestinatario, $peso, $volumen);
                    $total = $coberturaPorProducto->TotalTasacion->Total;
                }
                catch(Exception $e) {
                    $error = 'Ocurrio un error intentando calcular el costo de envío.';
                    $total = 0;
                }
                if($error !== '' && !wc_has_notice($error, 'error')){
                    wc_add_notice($error, 'error');
                }
                return $total;
            }

            function correoschile_calcular_coberturaPorProductoPorTienda($comunaRemitente, $tipoDespacho, $package){
                $total = 0;
                $notice = '';
                $error = '';
                
                if (! in_array( 'dokan-lite/dokan.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {
                    $error = 'Necesitas tener instalado el plugin dokan lite para poder utilizar la tarificación en modo multitienda.';
                } else {
                    $tiendas = array();
                    foreach ($package['contents'] as $item_id => $values ) {
                        $producto = $values['data'];
                        $tienda = dokan_get_vendor_by_product($values['product_id']);
                        $vendor_data = get_user_meta($tienda->id, 'dokan_profile_settings');
                        $nombreTienda = $vendor_data[0]['store_name'];
                        $comunaTienda = $vendor_data[0]['address']['city'];
                        if(!isset($comunaTienda) || $comunaTienda[0] === '') 
                            $notice .=  ($notice !== '' ? '<br/>' : '') . 'La tienda ' . $nombreTienda[0] . ' no tiene su comuna definida.';
                        $login = $tienda->data->data->user_login;
                        if($tienda){
                            if(!array_key_exists($login, $tiendas)){
                                $tiendas[$login] = array(
                                    'peso' => wc_get_weight($producto->get_weight(), 'cm') * $values['quantity'],
                                    'volumen' => ((wc_get_dimension($producto->get_width() * $producto->get_length() * $producto->get_height(), 'cm'))/1000000) * $values['quantity'],
                                    'comuna' => $comunaTienda,
                                    'tienda' => $nombreTienda);
                            } else {
                                $tiendas[$login]['peso'] = $tiendas[$login]['peso'] + (wc_get_weight($producto->get_weight(), 'cm') * $values['quantity']);
                                $tiendas[$login]['volumen'] = $tiendas[$login]['volumen'] + (((wc_get_dimension($producto->get_width() * $producto->get_length() * $producto->get_height(), 'cm'))/1000000) * $values['quantity']);
                            }
                        }
                    }
                    $total = 0;
                    foreach ($tiendas as $tienda) { 
                        try {
                            $kilos = $tienda['peso'];
                            $volumen = $tienda['volumen'];
                            $comunaDestinatario = $tienda['comuna'];
                            $coberturaPorProducto = correoschile_coberturaPorProducto($tipoDespacho, $comunaRemitente, $comunaDestinatario, $kilos, $volumen);
                            $total += $coberturaPorProducto->TotalTasacion->Total;
                        }
                        catch(Exception $e) {
                            $error = 'Ocurrio un error intentando calcular el costo de envío.';
                            $total = 0;
                        }
                    }
                }
                if($notice !== '' && !wc_has_notice($notice, 'notice')) wc_add_notice($notice, 'notice');
                if($error !== '' && !wc_has_notice($error, 'error')) wc_add_notice($error, 'error');
                return $total;
            }
        }
        add_action('woocommerce_before_shipping_calculator', 'correoschile_before_shipping_calculator', 10, 1);
        add_action('woocommerce_checkout_before_customer_details', 'correoschile_checkout_before_customer_details', 10, 1);
        add_action('woocommerce_after_checkout_validation', 'correoschile_validar_finalizar_compra', 10, 1);
        add_action('woocommerce_thankyou', 'correoschile_pedido_recibido', 10, 1);
    }
}

function correoschile_queryStringToArray($queryString)
{
    $result = array();
    if(strpos($queryString, '='))
    {
        if(strpos($queryString, '?') !== false) 
        {
            $q = parse_url($queryString);
            $queryString = $q['query'];
        }
    } 
    else
    {
        return false;
    }
    foreach (explode('&', $queryString) as $couple) 
    {
        list ($key, $val) = explode('=', $couple);
        $result[$key] = $val;
    }
    return empty($result) ? false : $result;
}

function correoschile_add_shipping_method( $methods ) 
{
    $methods['correoschile_shipping_method'] = 'CorreosChile_Shipping_Method';
    return $methods;
}

/**
 * Inicializa el script correoschile_cart.js, proporcionando los parámetros faltante para su correcta carga.
 */
function correoschile_before_shipping_calculator() 
{
    $html = "";
    $html.= "<script>";
    $html.= "jQuery(document).ready(function(){";
    $html.= "new correoschile_cart({";
    $html.= "'tipoDespacho':'". trim(WC()->session->get('correoschile_shipping_tipoDespacho')) . "',";
    $html.= "'direccion':'". trim(WC()->session->get('correoschile_shipping_direccion')) . "'";            
    $html.= "});";
    $html.= "});";
    $html.= "</script>";
    echo $html;
}

/**
* Inicializa el script correoschile_checkout.js, proporcionando los parámetros faltante para su correcta carga.
 */
function correoschile_checkout_before_customer_details()
{
    $html = "";
    $ocultarCompanyField = get_option('correoschile_ocultarCompanyField', '');
    if($ocultarCompanyField === 'S')
    {
        $html.= '<style>';
        $html.= '#billing_company_field, #shipping_company_field { display: none; } ';
        $html.= '</style>';
    }
    $html.= "<script>";
    $html.= "jQuery(document).ready(function(){";
    $html.= "new correoschile_checkout({";
    $html.= "'shipping_tipoDespacho':'". trim(WC()->session->get('correoschile_shipping_tipoDespacho')) . "',";
    $html.= "'billing_tipoDespacho':'". trim(WC()->session->get('correoschile_billing_tipoDespacho')) . "'";
    $html.= "});";
    $html.= "});";
    $html.= "</script>";
    echo $html;
}

/**
 * Realiza la validación de la información de la dirección de envío antes de realizar el pedido.
 * Noficando el error al formulario.
 * 
 * @param array $posted Contiene el formulario completo de checkout.
 */
function correoschile_validar_finalizar_compra($posted)
{
    $packages = WC()->shipping->get_packages();
    $chosen_methods = WC()->session->get('chosen_shipping_methods');
    if(is_array($chosen_methods) && in_array('correoschile_shipping_method', $chosen_methods))
    {
        foreach($packages as $i => $package )
        {
            if ($chosen_methods[$i] != 'correoschile_shipping_method')
            continue;
        }
        $mensaje = '';
        $ship_to_different_address = (bool)$posted['ship_to_different_address'];
        
        $prefijo = 'billing';
        WC()->session->set('correoschile_ship_to_different_address', 'N');
        if($ship_to_different_address)
        {
            WC()->session->set('correoschile_ship_to_different_address', 'S');
            $prefijo = 'shipping';
        }

        $pais = $posted[$prefijo . '_country'];
        $region = $posted[$prefijo . '_state'];
        $comuna = $posted[$prefijo . '_city'];
        $direccion = $posted[$prefijo . '_address_1'];
        
        //EVALUAR PAÍS: SOLO PUEDE SER CHILE
        if(!isset($pais) || $pais === '') {
            $mensaje .= ($mensaje !== '' ? '<br/>' : '') . 'No se ha seleccionado país.';
        } else {
            if($pais !== 'CL') {
                $mensaje .= ($mensaje !== '' ? '<br/>' : '') . 'El país seleccionado solo puede ser Chile.';
            } else {
                //EVALUAR REGION: SOLO PUEDE SER UNA DE CORREOS DE CHILE
                if(!isset($region) || $region === ''){
                    $mensaje .= ($mensaje !== '' ? '<br/>' : '') . 'No se ha seleccionado región.';
                } else {
                    $regionObtenida = correoschile_obtenerRegion($region);
                    if(!isset($regionObtenida)) {
                        $mensaje .= ($mensaje !== '' ? '<br/>' : '') . 'La región seleccionada no es válida.';
                    } else {
                        //EVALUAR COMUNA: SOLO PUEDE SER UNA DE CORREOS DE CHILE
                        if(!isset($comuna) || $comuna === '') {
                            $mensaje .= ($mensaje !== '' ? '<br/>' : '') . 'No se ha seleccionado comuna.';
                        } else {
                            $comunaObtenida = correoschile_obtenerComuna($regionObtenida->Identificador, $comuna);
                            if(!isset($comunaObtenida)) {
                                $mensaje .= ($mensaje !== '' ? '<br/>' : '') . 'La comuna seleccionada no es válida.';
                            } else {
                                //EVALUAR DIRECCIÓN:
                                $tipoDespacho = WC()->session->get('correoschile_' . $prefijo .'_tipoDespacho');
                                switch($tipoDespacho){
                                    case '07': //EVALUAR NOMBRE DE SUCURSAL
                                        if(!isset($direccion) || $direccion === '') {
                                            $mensaje .= ($mensaje !== '' ? '<br/>' : '') . 'No se ha seleccionado la sucursal.';
                                        } else {
                                            $sucursalObtenida = correoschile_nombreDeSucursal($comunaObtenida->Iata, $direccion);
                                            if(!isset($sucursalObtenida)){
                                                $mensaje .= ($mensaje !== '' ? '<br/>' : '') . 'La sucursal seleccionada no es válida.';
                                            }
                                        }
                                        break;
                                    default: //EVALUAR CODIGO POSTAL. (SE ASUMIRA POR DEFECTO DESPACHO A DOMICILIO)
                                        if(!isset($direccion) || $direccion === '') {
                                            $mensaje .= ($mensaje !== '' ? '<br/>' : '') . 'No se ha ingresado la dirección.';
                                        } else {
                                            try {
                                                $normalizacion = correoschile_normalizacion($direccion, $comuna);
                                            } catch (Exception $e) {
                                                $mensaje .= ($mensaje !== '' ? '<br/>' : '') . 'No se ha podido determinar el código postal de su dirección, por favor ingrese su dirección correctamente.';
                                            }
                                        }
                                        break;
                                }
                            }
                            
                            //EVALUAR TARIFICACIÓN: EN CASO DE ENCONTRAR PRODUCTOS QUE NO SON TARIFICABLES.
                            foreach ($package['contents'] as $item_id => $values ) { 
                                $producto = $values['data'];
                                $peso = is_numeric($producto->get_weight()) ? wc_get_weight($producto->get_weight(), 'kg') : 0;
                                $width = is_numeric($producto->get_width()) ? $producto->get_width() : 0;
                                $length = is_numeric($producto->get_length()) ? $producto->get_length() : 0;
                                $height = is_numeric($producto->get_height()) ? $producto->get_height() : 0;
                                $volumen = wc_get_dimension($width * $length * $height, 'cm');
                                if($peso === 0 || $volumen === 0){
                                    $mensaje .=  ($mensaje !== '' ? '<br/>' : '') . 'El producto "' . $producto->get_name() . '" no tiene su peso o dimensiones definidos.';
                                } else {
                                    if($volumen > 0) {
                                        $volumen = $volumen / 1000000;
                                    }
                                } 
                                if($peso > 50 || $volumen > 1) {
                                    $mensaje .=  ($mensaje !== '' ? '<br/>' : '') . 'El producto "' . $producto->get_name() . '" excede el peso o dimensiones aceptadas por nuestros servicios de paquetería.';
                                }
                            }
                        }
                    }
                }
            }
        }
        
        if($mensaje !== '') {
            $mensaje = '<strong>CorreosChile error:</strong><br/>' . $mensaje;
            if(!wc_has_notice($mensaje, 'error')){
                wc_add_notice($mensaje, 'error');
            }
        }
    }
}

/**
 * @param string $order_id identificador del pedido
 */
function correoschile_pedido_recibido($order_id)
{
    //DETECTAR CUANDO SE HAYA REALIZADO UNA COMPRA CON CORREOS DE CHILE
    $chosen_methods = WC()->session->get('chosen_shipping_methods');
    if(is_array($chosen_methods) && in_array('correoschile_shipping_method', $chosen_methods))
    {
        $order = wc_get_order($order_id);
        $ship_to_different_address = WC()->session->get('correoschile_ship_to_different_address');
        $ship_to_different_address = !isset($ship_to_different_address) ? 'N' : $ship_to_different_address;
        $tipoDespacho = WC()->session->get('correoschile_billing_tipoDespacho');
        if($ship_to_different_address === 'S') 
        {
            $tipoDespacho = WC()->session->get('correoschile_shipping_tipoDespacho');
        }
        $kilos = 0;
        $volumen = 0;
        $items = $order->get_items();
        $pf = new WC_Product_Factory();
        foreach ($items as $item)
        {
            $producto = $pf->get_product($item['product_id']);
            $peso = is_numeric($producto->get_weight()) ? wc_get_weight($producto->get_weight(), 'kg') : 0;
            $width = is_numeric($producto->get_width()) ? $producto->get_width() : 0;
            $length = is_numeric($producto->get_length()) ? $producto->get_length() : 0;
            $height = is_numeric($producto->get_height()) ? $producto->get_height() : 0;
            $kilos = $kilos + $peso * $item['quantity'];
            $volumen = $volumen + (wc_get_dimension($width * $length * $height, 'cm')/1000000) * $item['quantity'];
        }
        $volumen = round($volumen, 3);
        correoschile_agregarEnvio($order_id, $tipoDespacho, $kilos, $volumen);
    }
}


